/*
 * Decompiled with CFR 0.152.
 */
package org.digitalmediaserver.cuelib.io;

import java.io.EOFException;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.StringReader;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.List;
import java.util.Locale;
import org.digitalmediaserver.cuelib.CueParser;
import org.digitalmediaserver.cuelib.CueSheet;
import org.digitalmediaserver.cuelib.FileData;
import org.digitalmediaserver.cuelib.Index;
import org.digitalmediaserver.cuelib.Position;
import org.digitalmediaserver.cuelib.TrackData;
import org.digitalmediaserver.cuelib.util.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FLACReader {
    private static final Logger LOGGER = LoggerFactory.getLogger(FLACReader.class);
    public static final byte STREAMINFO = 0;
    public static final byte PADDING = 1;
    public static final byte APPLICATION = 2;
    public static final byte SEEKTABLE = 3;
    public static final byte VORBIS_COMMENT = 4;
    public static final byte CUESHEET = 5;
    public static final byte PICTURE = 6;
    public static final String DEFAULT_FILENAME = "self.flac";
    protected final ReadableByteChannel byteChannel;
    protected final Path file;

    public static CueSheet getCueSheet(Path file) throws IOException {
        if (file == null) {
            return null;
        }
        try (SeekableByteChannel byteChannel = Files.newByteChannel(file, new OpenOption[0]);){
            CueSheet cueSheet = new FLACReader(byteChannel, file).extractCueSheet(null);
            return cueSheet;
        }
    }

    public static CueSheet getCueSheet(Path file, ByteBuffer buffer) throws IOException {
        if (file == null) {
            return null;
        }
        try (SeekableByteChannel byteChannel = Files.newByteChannel(file, new OpenOption[0]);){
            CueSheet cueSheet = new FLACReader(byteChannel, file).extractCueSheet(buffer);
            return cueSheet;
        }
    }

    public static CueSheet getCueSheet(ReadableByteChannel byteChannel) {
        return byteChannel == null ? null : new FLACReader(byteChannel).extractCueSheet(null);
    }

    public static CueSheet getCueSheet(ReadableByteChannel byteChannel, Path file) {
        return byteChannel == null ? null : new FLACReader(byteChannel, file).extractCueSheet(null);
    }

    public static CueSheet getCueSheet(ReadableByteChannel byteChannel, ByteBuffer buffer) {
        return byteChannel == null ? null : new FLACReader(byteChannel).extractCueSheet(buffer);
    }

    public static CueSheet getCueSheet(ReadableByteChannel byteChannel, ByteBuffer buffer, Path file) {
        return byteChannel == null ? null : new FLACReader(byteChannel, file).extractCueSheet(buffer);
    }

    public FLACReader(ReadableByteChannel byteChannel) {
        this(byteChannel, null);
    }

    public FLACReader(ReadableByteChannel byteChannel, Path file) {
        this.byteChannel = byteChannel;
        this.file = file;
    }

    public CueSheet extractCueSheet() {
        return this.extractCueSheet(null);
    }

    public CueSheet extractCueSheet(ByteBuffer buffer) {
        boolean selfAllocated;
        if (buffer == null) {
            selfAllocated = true;
            buffer = ByteBuffer.allocateDirect(3);
            buffer.limit(0);
        } else {
            selfAllocated = false;
        }
        buffer.order(ByteOrder.BIG_ENDIAN);
        try {
            this.ensureAvailable(buffer, 3);
            byte b = buffer.get();
            if (b == 102 && buffer.get() == 76 && buffer.get() == 97) {
                if (selfAllocated) {
                    buffer = ByteBuffer.allocateDirect(2048);
                    buffer.order(ByteOrder.BIG_ENDIAN);
                    buffer.limit(0);
                }
                this.ensureAvailable(buffer, 1);
                if (buffer.get() != 67) {
                    return null;
                }
                return this.findCueSheet(buffer);
            }
            if (b == 73 && buffer.get() == 68 && buffer.get() == 51) {
                if (selfAllocated) {
                    buffer = ByteBuffer.allocateDirect(4096);
                    buffer.order(ByteOrder.BIG_ENDIAN);
                    buffer.limit(0);
                }
                return this.skipID3v2(buffer) ? this.extractCueSheet(buffer) : null;
            }
            return null;
        }
        catch (IOException e) {
            if (this.file == null) {
                LOGGER.error("An error occurred while parsing cue sheet from FLAC metadata: {}", (Object)e.getMessage());
            } else {
                LOGGER.error("An error occurred while parsing cue sheet from FLAC metadata in \"{}\": {}", (Object)this.file, (Object)e.getMessage());
            }
            LOGGER.trace("", e);
            return null;
        }
    }

    protected CueSheet findCueSheet(ByteBuffer buffer) throws IOException {
        StreamInfo streamInfo = null;
        while (true) {
            CueSheetResult result;
            this.ensureAvailable(buffer, 4);
            byte blockHeader = buffer.get();
            int blockType = blockHeader & 0x7F;
            int size = (buffer.get() & 0xFF) << 16 | (buffer.get() & 0xFF) << 8 | buffer.get() & 0xFF;
            int read = 0;
            if (blockType == 0) {
                streamInfo = this.parseStreamInfoBlock(buffer);
                read = size;
            } else if (blockType == 5) {
                result = this.parseCuesheetBlock(buffer, streamInfo);
                if (result.getCueSheet() != null) {
                    if (LOGGER.isDebugEnabled()) {
                        if (this.file == null) {
                            LOGGER.debug("Parsed the following cue sheet from the FLAC CUESHEET block:\n{}", (Object)result.getCueSheet());
                        } else {
                            LOGGER.debug("Parsed the following cue sheet from the FLAC CUESHEET block in \"{}\":\n{}", (Object)this.file, (Object)result.getCueSheet());
                        }
                    }
                    return result.getCueSheet();
                }
                read = result.getReadLength();
            } else if (blockType == 4) {
                result = this.parseVorbisCommentBlock(buffer);
                if (result.getCueSheet() != null) {
                    if (LOGGER.isDebugEnabled()) {
                        if (this.file == null) {
                            LOGGER.debug("Parsed the following cue sheet from the FLAC VORBIS_COMMENT block:\n{}", (Object)result.getCueSheet());
                        } else {
                            LOGGER.debug("Parsed the following cue sheet from the FLAC VORBIS_COMMENT block in \"{}\":\n{}", (Object)this.file, (Object)result.getCueSheet());
                        }
                    }
                    return result.getCueSheet();
                }
                buffer.order(ByteOrder.BIG_ENDIAN);
                read = result.getReadLength();
            }
            if (FLACReader.isLastBlock(blockHeader)) {
                return null;
            }
            this.skip(buffer, size - read);
        }
    }

    protected StreamInfo parseStreamInfoBlock(ByteBuffer buffer) throws IOException {
        this.skip(buffer, 10L);
        this.ensureAvailable(buffer, 4);
        int pos = buffer.position();
        int sampleRate = (buffer.get(pos) & 0xFF) << 12 | (buffer.get(pos + 1) & 0xFF) << 4 | (buffer.get(pos + 2) & 0xF0) >> 4;
        int bitsPerSample = (buffer.get(pos + 2) & 1) << 4 | ((buffer.get(pos + 3) & 0xF0) >> 4) + 1;
        this.skip(buffer, 24L);
        return new StreamInfo(sampleRate, bitsPerSample);
    }

    protected CueSheetResult parseCuesheetBlock(ByteBuffer buffer, StreamInfo streamInfo) throws IOException {
        if (LOGGER.isDebugEnabled()) {
            if (this.file == null) {
                LOGGER.debug("Parsing FLAC CUESHEET block");
            } else {
                LOGGER.debug("Parsing FLAC CUESHEET block in \"{}\"", (Object)this.file);
            }
        }
        CueSheet result = new CueSheet(this.file);
        int read = 0;
        result.setCatalog(this.readString(buffer, StandardCharsets.US_ASCII, 128, true, true));
        read += 128;
        this.skip(buffer, 267L);
        read += 267;
        this.ensureAvailable(buffer, 1);
        int numTracks = buffer.get() & 0xFF;
        ++read;
        if (numTracks > 0) {
            String fileNameStr;
            Path fileName = this.file == null ? null : this.file.getFileName();
            String string = fileNameStr = fileName == null ? null : fileName.toString();
            if (Utils.isBlank(fileNameStr)) {
                fileNameStr = DEFAULT_FILENAME;
            }
            FileData fileData = new FileData(result, fileNameStr, "WAVE");
            result.getFileData().add(fileData);
            List<TrackData> tracks = fileData.getTrackData();
            for (int i = 0; i < numTracks; ++i) {
                boolean isAudio;
                this.ensureAvailable(buffer, 9);
                long trackOffsetSamples = buffer.getLong();
                read += 8;
                TrackData track = new TrackData(fileData, buffer.get() & 0xFF, "AUDIO");
                ++read;
                track.setIsrcCode(this.readString(buffer, StandardCharsets.US_ASCII, 12, true, true));
                read += 12;
                this.ensureAvailable(buffer, 1);
                boolean bl = isAudio = (buffer.get() & 0x80) == 0;
                if (!isAudio) {
                    track.setDataType("DATA");
                }
                ++read;
                this.skip(buffer, 13L);
                read += 13;
                this.ensureAvailable(buffer, 1);
                int numIndicies = buffer.get() & 0xFF;
                ++read;
                for (int j = 0; j < numIndicies; ++j) {
                    this.ensureAvailable(buffer, 12);
                    long offsetSamples = buffer.getLong();
                    int indexNo = buffer.get() & 0xFF;
                    this.skip(buffer, 3L);
                    read += 12;
                    Position position = new Position(trackOffsetSamples + offsetSamples, streamInfo.getSampleRate());
                    Index index = new Index(indexNo, position);
                    track.getIndices().add(index);
                }
                int trackNo = track.getNumber();
                if (trackNo == 0 || (!isAudio || trackNo == 170) && (isAudio || trackNo == 255)) continue;
                tracks.add(track);
            }
        }
        return new CueSheetResult(result.getAllTrackData().isEmpty() ? null : result, read);
    }

    protected CueSheetResult parseVorbisCommentBlock(ByteBuffer buffer) throws IOException {
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        int read = 0;
        this.ensureAvailable(buffer, 4);
        int length = buffer.getInt();
        read += 4;
        this.skip(buffer, length);
        read += length;
        this.ensureAvailable(buffer, 4);
        length = buffer.getInt();
        read += 4;
        byte[] commentID = new byte[9];
        for (int i = 0; i < length; ++i) {
            this.ensureAvailable(buffer, 4);
            int commentLength = buffer.getInt();
            read += 4 + commentLength;
            if (commentLength > 9) {
                this.ensureAvailable(buffer, 9);
                buffer.get(commentID);
                if (!"CUESHEET=".equals(new String(commentID, StandardCharsets.US_ASCII).toUpperCase(Locale.ROOT))) {
                    this.skip(buffer, commentLength - 9);
                    continue;
                }
                if (LOGGER.isDebugEnabled()) {
                    if (this.file == null) {
                        LOGGER.debug("Parsing cue sheet from the FLAC VORBIS_COMMENTCUESHEET block");
                    } else {
                        LOGGER.debug("Parsing cue sheet from the FLAC VORBIS_COMMENTCUESHEET block in \"{}\"", (Object)this.file);
                    }
                }
                try (LineNumberReader reader = new LineNumberReader(new StringReader(this.readString(buffer, StandardCharsets.UTF_8, commentLength - 9, false, false)));){
                    Path fileName;
                    CueSheet result = CueParser.parse(reader, this.file);
                    if (this.file != null && (fileName = this.file.getFileName()) != null) {
                        String fileNameStr = fileName.toString();
                        for (FileData fileData : result.getFileData()) {
                            fileData.setFile(fileNameStr);
                        }
                    }
                    CueSheetResult cueSheetResult = new CueSheetResult(result, read);
                    return cueSheetResult;
                }
            }
            this.skip(buffer, commentLength);
        }
        return new CueSheetResult(null, read);
    }

    protected boolean skipID3v2(ByteBuffer buffer) throws IOException {
        this.ensureAvailable(buffer, 7);
        if (buffer.get() == -1 || buffer.get() == -1) {
            return false;
        }
        this.skip(buffer, 1L);
        byte[] bytes = new byte[4];
        buffer.get(bytes);
        int tagSize = ((bytes[0] & 0xFF) << 21) + ((bytes[1] & 0xFF) << 14) + ((bytes[2] & 0xFF) << 7) + (bytes[3] & 0xFF);
        this.skip(buffer, tagSize);
        return true;
    }

    protected void ensureAvailable(ByteBuffer buffer, int length) throws IOException {
        if (length > buffer.capacity()) {
            throw new IllegalStateException("Impossible to make " + length + " bytes available in a buffer of size " + buffer.capacity());
        }
        int available = buffer.remaining();
        if (available >= length) {
            return;
        }
        buffer.compact();
        while (available < length) {
            int count = this.byteChannel.read(buffer);
            if (count < 0) {
                throw new EOFException("The required number of bytes (" + length + " ) isn't available");
            }
            available += count;
        }
        buffer.flip();
    }

    protected void skip(ByteBuffer buffer, long count) throws IOException {
        if (count < 1L) {
            return;
        }
        if (count <= (long)buffer.remaining()) {
            buffer.position(buffer.position() + (int)count);
            return;
        }
        if (this.byteChannel instanceof SeekableByteChannel) {
            SeekableByteChannel seekable = (SeekableByteChannel)this.byteChannel;
            seekable.position(seekable.position() - (long)buffer.remaining() + count);
            buffer.position(0);
            buffer.limit(0);
            return;
        }
        long remainingSkip = count;
        while (remainingSkip > 0L) {
            if (remainingSkip <= (long)buffer.remaining()) {
                buffer.position(buffer.position() + (int)remainingSkip);
                return;
            }
            if (buffer.remaining() > 0) {
                remainingSkip -= (long)buffer.remaining();
                buffer.position(buffer.limit());
            }
            if (remainingSkip <= 0L) continue;
            buffer.clear();
            if (this.byteChannel.read(buffer) == -1) {
                throw new EOFException("The required number of bytes (" + count + " ) isn't available");
            }
            buffer.flip();
        }
    }

    protected String readString(ByteBuffer buffer, Charset charset, int byteLength, boolean nullTerminated, boolean returnNull) throws IOException {
        if (byteLength < 1) {
            return returnNull ? null : "";
        }
        byte[] bytes = new byte[byteLength];
        int pos = 0;
        int remainingBytes = byteLength;
        while (remainingBytes > 0) {
            if (remainingBytes <= buffer.remaining()) {
                buffer.get(bytes, pos, remainingBytes);
                break;
            }
            if (buffer.remaining() > 0) {
                int available = buffer.remaining();
                remainingBytes -= available;
                buffer.get(bytes, pos, available);
                pos += available;
            }
            if (remainingBytes <= 0) continue;
            buffer.clear();
            this.byteChannel.read(buffer);
            buffer.flip();
        }
        if (nullTerminated) {
            for (int i = bytes.length; i > 0; --i) {
                if (bytes[i - 1] == 0) continue;
                return new String(bytes, 0, i, charset);
            }
            return returnNull ? null : "";
        }
        return new String(bytes, charset);
    }

    protected static boolean isLastBlock(byte blockHeader) {
        return blockHeader < 0;
    }

    protected static class CueSheetResult {
        private final CueSheet cueSheet;
        private final int readLength;

        public CueSheetResult(CueSheet cueSheet, int readLength) {
            this.cueSheet = cueSheet;
            this.readLength = readLength;
        }

        public CueSheet getCueSheet() {
            return this.cueSheet;
        }

        public int getReadLength() {
            return this.readLength;
        }
    }

    protected static class StreamInfo {
        private final int sampleRate;
        private final int bitsPerSample;

        public StreamInfo(int sampleRate, int bitsPerSample) {
            this.sampleRate = sampleRate;
            this.bitsPerSample = bitsPerSample;
        }

        public int getSampleRate() {
            return this.sampleRate;
        }

        public int getBitsPerSample() {
            return this.bitsPerSample;
        }

        public String toString() {
            StringBuilder builder = new StringBuilder();
            builder.append("StreamInfo [sampleRate=").append(this.sampleRate).append(", bitsPerSample=").append(this.bitsPerSample).append("]");
            return builder.toString();
        }
    }
}

