/*
 * Decompiled with CFR 0.152.
 */
package net.pms.database;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Properties;
import net.pms.PMS;
import net.pms.configuration.UmsConfiguration;
import net.pms.database.Database;
import net.pms.gui.GuiManager;
import net.pms.util.UMSUtils;
import org.apache.commons.io.FileUtils;
import org.h2.engine.Constants;
import org.h2.tools.ConvertTraceFile;
import org.h2.tools.Upgrade;
import org.h2.util.Profiler;
import org.h2.util.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatabaseEmbedded {
    private static final Logger LOGGER = LoggerFactory.getLogger(DatabaseEmbedded.class);
    private static final UmsConfiguration CONFIGURATION = PMS.getConfiguration();
    private static final Profiler PROFILER = new Profiler();
    private static boolean collecting = false;

    private DatabaseEmbedded() {
    }

    public static String getJdbcUrl(String name) {
        DatabaseEmbedded.startCollectingIfNeeded();
        String dbDir = DatabaseEmbedded.getDbDir();
        String url = "jdbc:h2:" + dbDir + File.separator + name + ";DB_CLOSE_ON_EXIT=FALSE";
        LOGGER.info("Using database engine version {}.{}.{}", 2, 3, 232);
        int cacheSize = CONFIGURATION.getDatabaseMediaCacheSize();
        if (cacheSize < 0) {
            cacheSize = Utils.scaleForAvailableMemory(65536);
        }
        LOGGER.info("Database may use {} MB for caching", (Object)Math.round(cacheSize / 1024));
        url = url + ";CACHE_SIZE=" + cacheSize;
        if (CONFIGURATION.isDatabaseMediaUseCacheSoft()) {
            LOGGER.info("Database use soft cache");
            url = url + ";CACHE_TYPE=SOFT_LRU";
        } else {
            url = url + ";CACHE_TYPE=LRU";
        }
        if (CONFIGURATION.isDatabaseMediaUseMemoryIndexes()) {
            url = url + ";DEFAULT_TABLE_TYPE=MEMORY";
            LOGGER.info("Database indexes in memory is enabled");
        } else {
            url = url + ";DEFAULT_TABLE_TYPE=CACHED";
        }
        if (CONFIGURATION.getDatabaseLogging()) {
            url = url + ";TRACE_LEVEL_FILE=3";
            LOGGER.info("Database logging is enabled");
        } else if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Database logging is disabled");
        }
        LOGGER.debug("Using \"{}\" database URL: {}", (Object)name, (Object)url);
        LOGGER.info("Using \"{}\" database located at: \"{}\"", (Object)name, (Object)dbDir);
        return url;
    }

    private static String getDbDir() {
        File profileDirectory = new File(UmsConfiguration.getProfileDirectory());
        return new File(PMS.isRunningTests() || profileDirectory.isDirectory() ? UmsConfiguration.getProfileDirectory() : null, "database").getAbsolutePath();
    }

    public static String getDbUser() {
        return "sa";
    }

    public static String getDbPassword() {
        return null;
    }

    public static void shutdown(Connection connection) {
        DatabaseEmbedded.logProfilerIfNeeded();
        try (Statement stmt = connection.createStatement();){
            stmt.execute("SHUTDOWN COMPACT");
        }
        catch (SQLException e1) {
            LOGGER.error("compacting DB ", e1);
        }
    }

    public static boolean openFailed(String dbName, SQLException se) {
        String cause;
        String dbDir = DatabaseEmbedded.getDbDir();
        File dbFile = new File(dbDir + File.separator + dbName + ".data.db");
        File dbDirectory = new File(dbDir);
        if (se.getErrorCode() == 90048 && se.getMessage().contains("Unsupported database file version") && se.getCause() != null && se.getCause().getMessage() != null && (cause = se.getCause().getMessage()).contains("is smaller than the supported format")) {
            LOGGER.info("The database need a migration to the new h2 format");
            if (cause.contains("format 2 is smaller")) {
                DatabaseEmbedded.migrateDatabase(214, true, dbName);
                return true;
            }
            if (cause.contains("format 1 is smaller")) {
                DatabaseEmbedded.migrateDatabase(197, true, dbName);
                return true;
            }
        }
        if (se.getErrorCode() == 50000 && se.getMessage().contains("format 1 is smaller than the supported format")) {
            LOGGER.info("The database need a migration to the new h2 format");
            DatabaseEmbedded.migrateDatabase(197, true, dbName);
            return true;
        }
        if (dbFile.exists() || se.getErrorCode() == 90048) {
            FileUtils.deleteQuietly(dbDirectory);
            if (!dbDirectory.exists()) {
                LOGGER.info("The database has been deleted because it was corrupt or had the wrong version");
                return true;
            }
            Database.showMessageDialog("DamagedCacheCantBeDeleted", dbDir);
            LOGGER.error("Damaged cache can't be deleted. Stop the program and delete the folder \"" + dbDir + "\" manually");
            return false;
        }
        LOGGER.debug("Database connection error, retrying in 10 seconds");
        UMSUtils.sleep(10000);
        return true;
    }

    private static void migrateDatabase(int version, boolean deleteBackup, String dbName) {
        LOGGER.info("Migrating database to v{}", (Object)Constants.VERSION);
        GuiManager.setStatusLine("Migrating database to v" + Constants.VERSION);
        String dbDir = DatabaseEmbedded.getDbDir();
        String oldUrl = "jdbc:h2:" + dbDir + File.separator + dbName;
        Properties prprts = new Properties();
        String user = DatabaseEmbedded.getDbUser();
        String password = DatabaseEmbedded.getDbPassword();
        if (user != null) {
            prprts.setProperty("user", user);
        }
        if (password != null) {
            prprts.setProperty("password", password);
        }
        try {
            File dbBakFile;
            Upgrade.upgrade(oldUrl, prprts, version);
            if (deleteBackup && (dbBakFile = new File(dbDir + File.separator + dbName + ".mv.db.bak")).exists()) {
                dbBakFile.delete();
            }
            LOGGER.info("The database successfully migrated to version {}", (Object)Constants.VERSION);
        }
        catch (Exception e) {
            LOGGER.error("Database migration failed: {}", (Object)e.getMessage());
            LOGGER.trace("", e);
        }
    }

    public static void deleteDatabaseLock(String dbName) {
        String dbDir = DatabaseEmbedded.getDbDir();
        File dbLockFile = new File(dbDir + File.separator + dbName + ".lock.db");
        if (dbLockFile.exists()) {
            dbLockFile.delete();
        }
    }

    public static String getDatabaseFilename(String dbName) {
        String dbDir = DatabaseEmbedded.getDbDir();
        if (dbName == null || dbDir == null) {
            return null;
        }
        return dbDir + File.separator + dbName;
    }

    public static void createDatabaseReport(String dbName) {
        try {
            String dbFilename = DatabaseEmbedded.getDatabaseFilename(dbName);
            ConvertTraceFile.main("-traceFile", dbFilename + ".trace.db", "-script", dbFilename + "_logging_report.txt");
        }
        catch (SQLException ex) {
            LOGGER.trace("Failed to create trace database logging report");
        }
    }

    public static void startCollectingIfNeeded() {
        if (CONFIGURATION.getDatabaseLogging() && !collecting) {
            collecting = true;
            PROFILER.startCollecting();
        }
    }

    public static void logProfilerIfNeeded() {
        if (CONFIGURATION.getDatabaseLogging() && collecting) {
            collecting = false;
            LOGGER.trace("-------------------------------------------------------------");
            LOGGER.trace(PROFILER.getTop(5));
            LOGGER.trace("-------------------------------------------------------------");
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static void checkTableStorageType(Connection connection, String dbName) {
        try {
            boolean memoryStorage = CONFIGURATION.isDatabaseMediaUseMemoryIndexes();
            String askedStorageType = memoryStorage ? "MEMORY" : "CACHED";
            boolean upgradeTables = false;
            try (ResultSet rs = connection.getMetaData().getTables(null, "PUBLIC", "%", new String[]{"BASE TABLE"});){
                while (rs.next()) {
                    String tableName = rs.getString("TABLE_NAME");
                    PreparedStatement statement = connection.prepareStatement("SELECT STORAGE_TYPE FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_NAME = ? LIMIT 1");
                    try {
                        statement.setString(1, tableName);
                        ResultSet rs2 = statement.executeQuery();
                        try {
                            String storageType;
                            if (!rs2.first() || askedStorageType.equals(storageType = rs2.getString("STORAGE_TYPE"))) continue;
                            upgradeTables = true;
                            break;
                        }
                        finally {
                            if (rs2 == null) continue;
                            rs2.close();
                        }
                    }
                    finally {
                        if (statement == null) continue;
                        statement.close();
                    }
                }
            }
            if (!upgradeTables) return;
            LOGGER.info("Database will update to {} table storage type", (Object)askedStorageType);
            try (Statement stBackup = connection.createStatement();){
                boolean hasError = false;
                ResultSet backup = stBackup.executeQuery("SCRIPT NOPASSWORDS NOSETTINGS NOVERSION DROP");
                try (Statement stRestore = connection.createStatement();){
                    while (backup.next()) {
                        String sql = backup.getString(1);
                        if (!memoryStorage && sql.startsWith("CREATE MEMORY TABLE")) {
                            sql = sql.replace("CREATE MEMORY TABLE", "CREATE CACHED TABLE");
                        } else if (memoryStorage && sql.startsWith("CREATE CACHED TABLE")) {
                            sql = sql.replace("CREATE CACHED TABLE", "CREATE MEMORY TABLE");
                        }
                        try {
                            stRestore.execute(sql);
                        }
                        catch (SQLException e) {
                            hasError = true;
                        }
                    }
                }
                if (hasError) {
                    LOGGER.info("Database update error, backup file will be created");
                    String dbFilename = DatabaseEmbedded.getDatabaseFilename(dbName);
                    String backupFilename = dbFilename + ".script.sql";
                    File backupFile = new File(backupFilename);
                    try (BufferedWriter fileStream = new BufferedWriter(new FileWriter(backupFile));){
                        backup.beforeFirst();
                        while (backup.next()) {
                            String sql = backup.getString(1);
                            if (!memoryStorage && sql.startsWith("CREATE MEMORY TABLE")) {
                                sql = sql.replace("CREATE MEMORY TABLE", "CREATE CACHED TABLE");
                            } else if (memoryStorage && sql.startsWith("CREATE CACHED TABLE")) {
                                sql = sql.replace("CREATE CACHED TABLE", "CREATE MEMORY TABLE");
                            }
                            fileStream.write(sql);
                            fileStream.newLine();
                        }
                        fileStream.flush();
                        return;
                    }
                    catch (IOException iOException) {
                        return;
                    }
                }
                LOGGER.info("Database updated successfully");
                return;
            }
        }
        catch (SQLException ex) {
            LOGGER.error("Database error on memory indexes change", ex);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static int getCacheSize(Connection connection) {
        try (PreparedStatement statement = connection.prepareStatement("SELECT SETTING_VALUE FROM INFORMATION_SCHEMA.SETTINGS WHERE SETTING_NAME = 'info.CACHE_SIZE'");
             ResultSet result = statement.executeQuery();){
            if (!result.first()) return 0;
            int n = result.getInt("SETTING_VALUE");
            return n;
        }
        catch (SQLException ex) {
            LOGGER.error("Database error on getting memory cache size", ex);
        }
        return 0;
    }

    public static void analyzeDb(Connection connection) {
        try (PreparedStatement statement = connection.prepareStatement("ANALYZE SAMPLE_SIZE 0");){
            statement.execute();
        }
        catch (SQLException ex) {
            LOGGER.error("Database error on updating the selectivity statistics of tables", ex);
        }
    }
}

