# SPDX-FileCopyrightText: Copyright (c) 2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
# SPDX-License-Identifier: LicenseRef-NvidiaProprietary
#
# NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
# property and proprietary rights in and to this material, related
# documentation and any modifications thereto. Any use, reproduction,
# disclosure or distribution of this material and related documentation
# without an express license agreement from NVIDIA CORPORATION or
# its affiliates is strictly prohibited.

import subprocess

from nsys_recipe import nsys_constants
from nsys_recipe.lib import nsys_path
from nsys_recipe.log import logger


def _get_export_args(tables, format_type, output_path=None):
    export_args = [
        "--type",
        format_type,
        "--force-overwrite",
        "true",
        "--lazy",
        "false",
    ]

    if tables:
        export_args += ["--tables", ",".join(tables)]

    if output_path is not None:
        export_args += ["--output", output_path]

    return export_args


def export_file(report_path, tables=None, format_type="parquetdir", output_path=None):
    """Export report file to a specified format.

    This function overwrites any existing export files with the same name.

    Parameters
    ----------
    tables : str or list of str, optional
        Table(s) to export. If not given, all tables will be exported.
    format_type : str, optional
        The export file format. Default is 'parquetdir'.
    output_path : str, optional
        The export file path. Default is the current directory.

    Returns
    -------
    result : bool
        True if export succeeded, False otherwise.
    """
    # TODO(DTSP-15985): Support directory of files for SQLite.
    # We do not currently offer support for partial exports in SQLite.
    # Regardless of the 'table' argument, all tables will always be imported.
    if format_type == "sqlite":
        tables = None

    if isinstance(tables, str):
        tables = [tables]

    export_args = _get_export_args(tables, format_type, output_path)

    nsys_exe = nsys_path.find_installed_file(nsys_constants.NSYS_EXE_NAME)
    cmd = [nsys_exe, "export", *export_args, str(report_path)]

    tables_str = f" {tables}" if tables else ""
    output_path_str = f" to {output_path}" if output_path else ""

    logger.info(f"{report_path}: Exporting{tables_str}{output_path_str}...")
    logger.debug(f"{report_path}: Running {' '.join(cmd)}")

    try:
        p = subprocess.run(cmd, text=True, capture_output=True)
    except Exception as e:
        logger.error(f"{report_path}: Failed to export report: {e}")
        return False

    if p.returncode:
        logger.error(f"{report_path}: Failed to export report: {p.stderr}")
        return False

    return True
