/*
 * Copyright (c) BindView Development Corporation, 2001
 * See LICENSE file.
 * Author: Todd Sabin <tsabin@razor.bindview.com>
 */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include "samr.h"

void Usage (char * pszProgramName)
{
    printf ("Usage: %s [options] <target>\n", pszProgramName);
    printf (" options:\n");
    printf ("   -p protocol_sequence\n");
    printf ("   -e endpoint\n");
    printf (" examples:\n");
    printf ("   walksam 192.168.1.1\n");
    printf ("   walksam -p ncacn_ip_tcp -e 1054 192.168.1.2\n");
    exit (1);
}

void
print_nttime (char *label, FILETIME *ft)
{
    SYSTEMTIME st = { 0 };

    if (((ft->dwLowDateTime == 0) && (ft->dwHighDateTime == 0))
        || ((ft->dwLowDateTime == 0xffffffff) && (ft->dwHighDateTime == 0x7fffffff))) {
        printf ("%s never\n", label);
    } else {
        FileTimeToSystemTime (ft, &st);
        printf ("%s %d/%d/%d %d:%02d:%02d.%d\n", label, st.wMonth, st.wDay, st.wYear,
                st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
    }
}


void
print_uni_string (char *label, LSA_UNICODE_STRING *uni_string)
{
    char buffer[500] = "";

    WideCharToMultiByte (CP_ACP, 0, uni_string->Buffer,
                         uni_string->Length / 2,
                         buffer, sizeof (buffer),
                         NULL, NULL);
    printf ("%s%s\n", label, buffer);
}


void
do_user (SAM_HANDLE hUsr, PLSA_SID pDomSid)
{
    NTSTATUS rc;
    PSAM_USER_Q_INFO pInfo;

    pInfo = NULL;
    rc = SamrQueryInfoUser (hUsr, 21, &pInfo);
    if (rc >= 0) {
        print_uni_string ("Userid: ", &pInfo->tagged_union.Lvl21.userid);
        print_uni_string ("Full Name: ", &pInfo->tagged_union.Lvl21.full_name);
        print_uni_string ("Home Dir: ", &pInfo->tagged_union.Lvl21.home_dir);
        print_uni_string ("Home Drive: ", &pInfo->tagged_union.Lvl21.home_drive);
        print_uni_string ("Logon Script: ", &pInfo->tagged_union.Lvl21.logon_script);
        print_uni_string ("Profile: ", &pInfo->tagged_union.Lvl21.profile);
        print_uni_string ("Description: ", &pInfo->tagged_union.Lvl21.description);
        print_uni_string ("Workstations: ", &pInfo->tagged_union.Lvl21.workstations);
        print_uni_string ("Profile: ", &pInfo->tagged_union.Lvl21.profile);
        print_uni_string ("User Comment: ", &pInfo->tagged_union.Lvl21.user_comment);
        print_nttime ("Last Logon: ", (FILETIME *)&pInfo->tagged_union.Lvl21.last_logon);
        print_nttime ("Last Logoff: ", (FILETIME *)&pInfo->tagged_union.Lvl21.last_logoff);
        print_nttime ("Last Passwd Change: ", (FILETIME *)&pInfo->tagged_union.Lvl21.last_pwd_change);
        print_nttime ("Acct. Expires: ", (FILETIME *)&pInfo->tagged_union.Lvl21.acct_expiry);
        print_nttime ("Allowed Passwd Change: ", (FILETIME *)&pInfo->tagged_union.Lvl21.allow_pwd_change);

        printf ("Rid: %d\n", pInfo->tagged_union.Lvl21.Rid);
        printf ("Primary Group Rid: %d\n", pInfo->tagged_union.Lvl21.PrimaryGroupRid);
        printf ("Flags: 0x%x\n", pInfo->tagged_union.Lvl21.AcctFlags);
        printf ("Fields Present: 0x%x\n", pInfo->tagged_union.Lvl21.fields_present);
        printf ("Bad Password Count: %d\n", pInfo->tagged_union.Lvl21.bad_pwd_count);
        printf ("Num Logons: %d\n", pInfo->tagged_union.Lvl21.num_logons);
    } else {
        printf ("SamrQueryInfoUser(21) failed: 0x%x\n", rc);
    }
    printf ("\n");
}

void
walk_one_domain (SAM_HANDLE hDom, PLSA_SID pSid, ULONG start_rid)
{
    NTSTATUS rc;
    ULONG rid;
    ULONG consec_errors = 0;
    SAM_HANDLE hUsr = 0, hGrp = 0, hAls = 0;

    rc = 0;
    for (rid = start_rid; ((rc == 0) || ((rc == 0xc0000073) && (consec_errors++ < 100))); rid++) {
        SAM_TRANS_NAMES names = { 0, 0 };
        SAM_TRANS_USES  uses = { 0, 0 };

        rc = SamrLookupIds (hDom, 1, &rid, &names, &uses);
        if (rc >= 0) {
            char name[256] = "";

            consec_errors = 0;

            if (names.Count == 1) {
                WideCharToMultiByte (CP_ACP, 0, names.Names->Buffer,
                                     names.Names->Length / 2,
                                     name, sizeof (name),
                                     NULL, NULL);
            }

            switch (uses.Uses[0]) {
            case 1:
                printf ("rid %d: user %s\n", rid, name);
                rc = SamrOpenUser (hDom, 0x02000000, rid, &hUsr);
                if (rc >= 0) {
                    do_user (hUsr, pSid);
                    SamrClose (&hUsr);
                } else {
                    printf ("SamrOpenUser failed: 0x%x\n", rc);
                }
                break;

            case 2:
                printf ("rid %d: group %s\n", rid, name);
                break;

            case 4:
                printf ("rid %d: alias %s\n", rid, name);
                break;

            default:
                printf ("rid %d: use: %d\n", rid, uses.Uses[0]);
                break;
            }
        }
    }
}

int
do_domain (SAM_HANDLE hnd, SAM_DOMAIN *pDom)
{
    NTSTATUS rc;
    PLSA_SID pSid = NULL;

    rc = SamrLookupDomain (hnd, &pDom->Name, &pSid);
    if (rc >= 0) {
        SAM_HANDLE hDom;

        rc = SamrOpenDomain (hnd, 0x02000000, pSid, &hDom);
        if (rc >= 0) {
            walk_one_domain (hDom, pSid, 500);
            walk_one_domain (hDom, pSid, 1000);
            SamrClose (&hDom);
        } else {
            printf ("SamrOpenDomain failed: 0x%x\n", rc);
        }
    } else {
        printf ("SamrLookupDomain failed: 0x%x\n", rc);
    }
    return 0;
}


int
main (int argc, char **argv)
{
    RPC_STATUS rpcErr;
    char *protseq = "ncacn_np";
    char *target = NULL;
    char *endpoint = "\\pipe\\samr";
    char *strBinding = NULL;
    handle_t hello_IfHandle;
    SEC_WINNT_AUTH_IDENTITY ident = { (char *)L"", 0,
                                      (char *)L"", 0,
                                      (char *)L"", 0,
                                      SEC_WINNT_AUTH_IDENTITY_UNICODE };
    int i;

    for (i=1; i<argc; i++) {
        if (argv[i][0] == '-') {
            switch (argv[i][1]) {
            case 'p':
                protseq = argv[++i];
                break;

            case 'e':
                endpoint = argv[++i];
                break;

            default:
                Usage (argv[0]);
            }
        } else {
            target = argv[i];
        }
    }

    if (!target) {
        Usage (argv[0]);
    }

    rpcErr = RpcStringBindingCompose (NULL, protseq, target,
                                      endpoint, NULL, &strBinding);
    if (rpcErr != RPC_S_OK) {
        printf ("RpcStringBindingCompose returned 0x%x\n", rpcErr);
        exit (1);
    }

    rpcErr = RpcBindingFromStringBinding(strBinding, &hello_IfHandle);
    if (rpcErr != RPC_S_OK) {
        printf ("RpcBindingFromStringBinding returned 0x%x\n", rpcErr);
        exit (1);
    }
    
    if (strcmp (protseq, "ncacn_np") != 0) {
        rpcErr = RpcBindingSetAuthInfo (hello_IfHandle, NULL,
                                        RPC_C_AUTHN_LEVEL_CALL,
                                        RPC_C_AUTHN_WINNT,
                                        &ident, 0);
        if (rpcErr) {
            printf ("RpcBindingSetAuthInfo failed: 0x%x\n", rpcErr);
            exit (1);
        }
    }

    RpcTryExcept {
        long rc;
        SAM_HANDLE hnd;
        wchar_t *system = L"\\";
        ULONG resume = 0, numret = 0;
        PSAM_ENUM_DOMAINS pDomains = NULL;

        rc = SamrConnect2 (hello_IfHandle, &system, 0x02000000, &hnd);
        if (rc >= 0) {
            resume = 0;
            rc = SamrEnumDomains (hnd, &resume, 200, &pDomains, &numret);
            if (rc >= 0) {
                unsigned int i;
                for (i=0; i<numret; i++) {
                    do_domain (hnd, &pDomains->Domains[i]);
                }
            } else {
                printf ("SamrEnumDomains failed: 0x%x\n", rc);
            }
            rc = SamrClose (&hnd);
        } else {
            printf ("SamrConnect2 failed: 0x%x\n", rc);
        }
    } RpcExcept (1) {
        long exc = RpcExceptionCode ();
        printf ("Runtime reported exception 0x%x = %d\n", exc, exc);
    }
    RpcEndExcept;

    RpcStringFree (&strBinding);

    RpcBindingFree (&hello_IfHandle);
}


void __RPC_FAR * __RPC_USER
midl_user_allocate (size_t len)
{
    return calloc (1, len);
}

void __RPC_USER
midl_user_free (void __RPC_FAR *ptr)
{
    free (ptr);
}
