### BEGIN LICENSE ###
### Use of the triage tools and related source code is subject to the terms
### of the license below.
###
### ------------------------------------------------------------------------
### Copyright (C) 2011 Carnegie Mellon University. All Rights Reserved.
### ------------------------------------------------------------------------
### Redistribution and use in source and binary forms, with or without
### modification, are permitted provided that the following conditions are
### met:
###
### 1. Redistributions of source code must retain the above copyright
###    notice, this list of conditions and the following acknowledgments
###    and disclaimers.
###
### 2. Redistributions in binary form must reproduce the above copyright
###    notice, this list of conditions and the following disclaimer in the
###    documentation and/or other materials provided with the distribution.
###
### 3. All advertising materials for third-party software mentioning
###    features or use of this software must display the following
###    disclaimer:
###
###    "Neither Carnegie Mellon University nor its Software Engineering
###     Institute have reviewed or endorsed this software"
###
### 4. The names "Department of Homeland Security," "Carnegie Mellon
###    University," "CERT" and/or "Software Engineering Institute" shall
###    not be used to endorse or promote products derived from this software
###    without prior written permission. For written permission, please
###    contact permission@sei.cmu.edu.
###
### 5. Products derived from this software may not be called "CERT" nor
###    may "CERT" appear in their names without prior written permission of
###    permission@sei.cmu.edu.
###
### 6. Redistributions of any form whatsoever must retain the following
###    acknowledgment:
###
###    "This product includes software developed by CERT with funding
###     and support from the Department of Homeland Security under
###     Contract No. FA 8721-05-C-0003."
###
### THIS SOFTWARE IS PROVIDED BY CARNEGIE MELLON UNIVERSITY ``AS IS'' AND
### CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
### EXPRESS OR IMPLIED, AS TO ANY MATTER, AND ALL SUCH WARRANTIES, INCLUDING
### WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ARE
### EXPRESSLY DISCLAIMED. WITHOUT LIMITING THE GENERALITY OF THE FOREGOING,
### CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND
### RELATING TO EXCLUSIVITY, INFORMATIONAL CONTENT, ERROR-FREE OPERATION,
### RESULTS TO BE OBTAINED FROM USE, FREEDOM FROM PATENT, TRADEMARK AND
### COPYRIGHT INFRINGEMENT AND/OR FREEDOM FROM THEFT OF TRADE SECRETS.
### END LICENSE ###
'''
Contains a list of rules used to classify the state of a GDB Inferior.
Rules are defined by category (ex: "EXPLOITABLE") and are roughly ordered
from "most exploitable" to "least exploitable".
'''

from analyzers import *

rules = [
('EXPLOITABLE', [
    dict(analyzer=isReturnAv,
    desc="Access violation during return instruction",
    short_desc="ReturnAv",
    explanation="The target crashed on a return instruction, which likely "
    "indicates stack corruption."),

    dict(analyzer=isSegFaultOnPcNotNearNull,
    desc="Segmentation fault on program counter",
    short_desc="SegFaultOnPc",
    explanation="The target tried to access data at an address that matches "
    "the program counter. This is likely due to the execution of a branch "
    "instruction (ex: 'call') with a bad argument, but it could also be "
    "due to execution continuing past the end of a memory region or another "
    "cause. Regardless this likely indicates that the program counter "
    "contents are tainted and can be controlled by an attacker."),

    dict(analyzer=isBranchAvNotNearNull,
    desc="Access violation during branch instruction",
    short_desc="BranchAv",
    explanation="The target crashed on a branch instruction, which may "
    "indicate that the control flow is tainted."),

    dict(analyzer=isErrorWhileExecutingFromStack,
    desc="Executing from stack",
    short_desc="StackCodeExection",
    explanation="The target stopped on an error while executing code "
    "within the process's stack region."),

    dict(analyzer=isStackBufferOverflow,
    desc="Stack buffer overflow",
    short_desc="StackBufferOverflow",
    explanation="The target stopped while handling a signal that was "
    "generated by libc due to detection of a stack buffer overflow. Stack "
    "buffer overflows are generally considered exploitable."),

    dict(analyzer=isPossibleStackCorruption,
    desc="Possible stack corruption",
    short_desc="PossibleStackCorruption",
    explanation="GDB generated an error while unwinding the stack "
    "and/or the stack contained return addresses that were not mapped "
    "in the inferior's process address space and/or the stack pointer is "
    "pointing to a location outside the default stack region. These "
    "conditions likely indicate stack corruption, which is generally "
    "considered exploitable."),

    dict(analyzer=isDestAvNotNearNull,
    desc="Access violation on destination operand",
    short_desc="DestAv",
    explanation="The target crashed on an access violation at an address "
    "matching the destination operand of the instruction. This likely "
    "indicates a write access violation, which means the attacker may "
    "control the write address and/or value."),

    dict(analyzer=isMalformedInstructionSignal,
    desc="Bad instruction",
    short_desc="BadInstruction",
    explanation="The target tried to execute a malformed or privileged "
    "instruction. This may indicate that the control flow is tainted."),

    dict(analyzer=isHeapError,
    desc="Heap error",
    short_desc="HeapError",
    explanation="The target's backtrace indicates that libc has detected "
    "a heap error or that the target was executing a heap function when it "
    "stopped. This could be due to heap corruption, passing a bad pointer to "
    "a heap function such as free(), etc. Since heap errors might include "
    "buffer overflows, use-after-free situations, etc. they are generally "
    "considered exploitable."),
]),

('PROBABLY_EXPLOITABLE', [
    dict(analyzer=isStackOverflow,
    desc="Stack overflow",
    short_desc="StackOverflow",
    explanation="The target crashed on an access violation where the faulting "
    "instruction's mnemonic and the stack pointer seem to indicate a stack "
    "overflow."),

    dict(analyzer=isSegFaultOnPcNearNull,
    desc="Segmentation fault on program counter near NULL",
    short_desc="SegFaultOnPcNearNull",
    explanation="The target tried to access data at an address that matches "
    "the program counter. This may indicate that the program "
    "counter contents are tainted, however, it may also indicate a simple "
    "NULL deference."),

    dict(analyzer=isBranchAvNearNull,
    desc="Access violation near NULL during branch instruction",
    short_desc="BranchAvNearNull",
    explanation="The target crashed on a branch instruction, which may "
    "indicate that the control flow is tainted. However, there is a "
    "chance it could be a NULL dereference."),

    dict(analyzer=isBlockMove,
    desc="Access violation during block move",
    short_desc="BlockMoveAv",
    explanation="The target crashed during a block move, which may indicate "
    "that the attacker can control a buffer overflow."),

    dict(analyzer=isDestAvNearNull,
    desc="Access violation near NULL on destination operand",
    short_desc="DestAvNearNull",
    explanation="The target crashed on an access violation at an address "
    "matching the destination operand of the instruction. This likely "
    "indicates a write access violation, which means the attacker may "
    "control write address and/or value. However, it there is a chance "
    "it could be a NULL dereference."),
]),

('PROBABLY_NOT_EXPLOITABLE', [
    dict(analyzer=isSourceAvNearNull,
    desc="Access violation near NULL on source operand",
    short_desc="SourceAvNearNull",
    explanation="The target crashed on an access violation at an address "
    "matching the source operand of the current instruction. This likely "
    "indicates a read access violation, which may mean the application "
    "crashed on a simple NULL dereference to data structure that has no "
    "immediate effect on control of the processor."),

    dict(analyzer=isFloatingPointException,
    desc="Floating point exception signal",
    short_desc="FloatingPointException",
    explanation="The target crashed on a floating point exception. This "
    "may indicate a division by zero or a number of other floating point "
    "errors. It is generally difficult to leverage these types of errors "
    "to gain control of the processor."),

    dict(analyzer=isBenignSignal,
    desc="Benign signal",
    short_desc="BenignSignal",
    explanation="The target is stopped on a signal that either does not "
    "indicate an error or indicates an error that is generally not "
    "considered exploitable."),
]),

('UNKNOWN', [
    dict(analyzer=isSourceAvNotNearNull,
    desc="Access violation on source operand",
    short_desc="SourceAv",
    explanation="The target crashed on an access violation at an address "
    "matching the source operand of the current instruction. This likely "
    "indicates a read access violation."),

    dict(analyzer=isAbortSignal,
    desc="Abort signal",
    short_desc="AbortSignal",
    explanation="The target is stopped on a SIGABRT. SIGABRTs are often "
    "generated by libc and compiled check-code to indicate potentially "
    "exploitable conditions. Unfortunately this command does not yet further "
    "analyze these crashes."),

    dict(analyzer=isAccessViolationSignal,
    desc="Access violation",
    short_desc="AccessViolation",
    explanation="The target crashed due to an access violation but there is not enough "
    "additional information available to determine exploitability."),

    dict(analyzer=isUncategorizedSignal,
    desc="Uncategorized signal",
    short_desc="UncategorizedSignal",
    explanation="The target is stopped on a signal. This may be an exploitable "
    "condition, but this command was unable to categorize it."),
]),

] # end rules
