"use strict";
var estraverse = require('estraverse');
var coverage_branch_1 = require('./coverage-branch');
var CoverageStatement = require('./coverage-statement');
var CoverageFunction = (function () {
    function CoverageFunction(key, name, originalAST, pathFile, coverageObjectName) {
        this.key = key;
        this.name = name;
        this.pathFile = pathFile;
        this.coverageObjectName = coverageObjectName;
        this.setParameters(originalAST);
    }
    CoverageFunction.prototype.setParameters = function (originalAST) {
        var that = this;
        this.parameters = null;
        estraverse.traverse(originalAST, {
            enter: function (node) {
                if (that.isFunctionToTest(node)) {
                    that.parameters = node.params;
                    this.break();
                }
            }
        });
        if (this.parameters === null) {
            throw new Error('Unable to get parameters of function "' + this.name + '"');
        }
    };
    CoverageFunction.prototype.setStartLocation = function (locationStart, locationEnd) {
        this.locationStartFunctionDeclaration = {};
        this.locationStartFunctionDeclaration.start = locationStart;
        this.locationStartFunctionDeclaration.end = locationEnd;
    };
    CoverageFunction.prototype.setEndLocation = function (originalAST) {
        var that = this;
        this.locationEndFunctionDeclaration = {};
        estraverse.traverse(originalAST, {
            enter: function (node) {
                if (that.isFunctionToTest(node)) {
                    if (node.body !== undefined && node.body.loc !== undefined) {
                        that.locationEndFunctionDeclaration.start = {
                            line: node.body.loc.start.line,
                            column: node.body.loc.start.column
                        };
                        that.locationEndFunctionDeclaration.end = {
                            line: node.body.loc.end.line,
                            column: node.body.loc.end.column
                        };
                        that.functionAST = node;
                        this.break();
                    }
                }
            }
        });
    };
    CoverageFunction.prototype.addStatements = function (statement, statementMap) {
        var startLocation;
        var endLocation;
        var stmtInstance;
        this.statements = [];
        for (var key in statementMap) {
            if (statementMap.hasOwnProperty(key)) {
                startLocation = {};
                startLocation.line = statementMap[key].start.line;
                startLocation.column = statementMap[key].start.column;
                endLocation = {};
                endLocation.line = statementMap[key].end.line;
                endLocation.column = statementMap[key].end.column;
                if (this.objectMapIsInsideFunction(startLocation, endLocation)) {
                    try {
                        stmtInstance = new CoverageStatement(key, startLocation, endLocation, this.functionAST);
                        stmtInstance.initializeValues(statement);
                        this.statements.push(stmtInstance);
                    }
                    catch (e) {
                        throw e;
                    }
                }
            }
        }
    };
    CoverageFunction.prototype.addBranches = function (branch, branchMap) {
        var startLocation;
        var endLocation;
        var branchInstance;
        var handleBranches = [
            'if',
            'switch',
            'cond-expr'
        ];
        this.branches = [];
        for (var key in branchMap) {
            if (branchMap.hasOwnProperty(key)) {
                if (handleBranches.indexOf(branchMap[key].type) === -1) {
                    continue;
                }
                startLocation = {};
                startLocation.line = branchMap[key].locations[0].start.line;
                startLocation.column = branchMap[key].locations[0].start.column;
                endLocation = {};
                endLocation.line = branchMap[key].locations[0].end.line;
                endLocation.column = branchMap[key].locations[0].end.column;
                if (this.objectMapIsInsideFunction(startLocation, endLocation)) {
                    try {
                        branchInstance = new coverage_branch_1.CoverageBranch(key, startLocation, endLocation, this.functionAST);
                        try {
                            branchInstance.initializeValues(branch);
                            this.branches.push(branchInstance);
                        }
                        catch (e) {
                            throw e;
                        }
                    }
                    catch (e) {
                        throw e;
                    }
                }
            }
        }
    };
    CoverageFunction.prototype.execute = function (coverageObject, cb) {
        var statements_ = coverageObject.s;
        var statementsMap = coverageObject.statementMap;
        var branches_ = coverageObject.b;
        var branchesMap = coverageObject.branchMap;
        var key;
        var found;
        var error;
        for (var k = 0; k < this.statements.length; k++) {
            found = false;
            for (key in statementsMap) {
                if (statementsMap.hasOwnProperty(key) && key == this.statements[k].key) {
                    try {
                        this.statements[k].updateValues(statements_);
                        found = true;
                        break;
                    }
                    catch (e) {
                        error = new Error('Unable to update values of statement ' + key + '. Reason: ' + e.message);
                        cb(error, null);
                    }
                }
            }
            if (!found) {
                error = new Error('Unable to update values of statement ' + key + '. Object not found.');
                cb(error, null);
            }
        }
        var indexSwitchStatement = 0;
        var numSwitchs = {
            'branches': 0,
            'statements': 0
        };
        for (var k = 0; k < this.branches.length; k++) {
            found = false;
            for (key in branchesMap) {
                if (branchesMap.hasOwnProperty(key) && key == this.branches[k].key) {
                    try {
                        this.branches[k].updateValues(branches_);
                        found = true;
                        if (this.branches[k].branchType === coverage_branch_1.BranchType.Switch) {
                            numSwitchs['branches']++;
                            for (var j = indexSwitchStatement; j < this.statements.length; j++) {
                                if (this.statements[j].type === 'SwitchStatement') {
                                    indexSwitchStatement = j + 1;
                                    numSwitchs['statements']++;
                                    break;
                                }
                            }
                        }
                        break;
                    }
                    catch (e) {
                        error = new Error('Unable to update values of branches ' + key + '. Reason: ' + e.message);
                        cb(error, null);
                    }
                }
            }
            if (!found) {
                error = new Error('Unable to update values of branches ' + key + '. Object not found.');
                cb(error, null);
            }
        }
        if (numSwitchs['branches'] !== numSwitchs['statements']) {
            error = new Error('Unable to update values of branches/statements ' +
                '(numSwitchs["branches"] !== numSwitchs["statements"])');
            cb(error, null);
        }
        cb(null, true);
    };
    CoverageFunction.prototype.objectMapIsInsideFunction = function (sl, el) {
        var locStartFunc = {};
        var locEndFunc = {};
        locStartFunc.line = this.locationStartFunctionDeclaration.start.line;
        locEndFunc.line = this.locationEndFunctionDeclaration.end.line;
        locEndFunc.column = this.locationEndFunctionDeclaration.end.column;
        if (sl.line >= locStartFunc.line) {
            if (el.line < locEndFunc.line) {
                return true;
            }
            else if (el.line === locEndFunc.line) {
                return el.column < locEndFunc.column;
            }
        }
        return false;
    };
    CoverageFunction.prototype.isFunctionToTest = function (node) {
        var isFunctionToTest_;
        isFunctionToTest_ = node.type === 'FunctionExpression' || node.type === 'FunctionDeclaration';
        isFunctionToTest_ = isFunctionToTest_ && node.id !== null;
        isFunctionToTest_ = isFunctionToTest_ && node.id.type === 'Identifier' && node.id.name === this.name;
        return isFunctionToTest_;
    };
    return CoverageFunction;
}());
module.exports = CoverageFunction;
