"use strict";
var fs = require('fs');
var path = require('path');
var chalk = require('chalk');
var mkdirp = require('mkdirp');
var logger = require('./logger');
var utils = require('./utils');
function handleEvent(event, pathFile, leenaConfig) {
    var ret = {};
    var message = 'Event on "' + pathFile + '" ' +
        '(' + chalk.green.underline(event) + ')';
    var commonPath;
    ret.pathFile = pathFile;
    ret.pathTempFile = null;
    ret.fileToTest = false;
    commonPath = pathFile.split(leenaConfig.browserSync.watcher.server);
    if (commonPath.length !== 2) {
        throw new Error('Unable to get the common path between application and temporary application');
    }
    commonPath = path.normalize(commonPath[1]);
    logger.info(message);
    switch (event) {
        case 'add':
            if (utils.fileExists(pathFile)) {
                ret = add(pathFile, commonPath, leenaConfig);
            }
            else {
                throw new Error('Unable to handle "add" event. "' + pathFile + '" does not exist');
            }
            break;
        case 'change':
            if (utils.fileExists(pathFile)) {
                ret = change(pathFile, commonPath, leenaConfig);
            }
            else {
                throw new Error('Unable to handle "change" event. "' + pathFile + '" does not exist');
            }
            break;
        case 'unlink':
            ret = unlink(pathFile, commonPath, leenaConfig);
            break;
        case 'unlinkDir':
            break;
    }
    return ret;
}
exports.handleEvent = handleEvent;
function add(pathFile, commonPath, leenaConfig) {
    var ret = {};
    var fileToTest = isFileToTest(pathFile, leenaConfig);
    var tempPathFile;
    var tempPath;
    var contentToWrite;
    tempPathFile = path.normalize([
        leenaConfig.browserSync.webServer.server,
        commonPath
    ].join(path.sep));
    tempPath = path.dirname(tempPathFile);
    if (!utils.pathExists(tempPath)) {
        mkdirp.sync(tempPath, { mode: 755 });
        if (!utils.pathExists(tempPath)) {
            throw new Error('Unable to create directories recursively');
        }
    }
    contentToWrite = utils.readFile(pathFile);
    if (contentToWrite === null) {
        throw new Error('Unable to read "' + pathFile + '" in the "add" handler');
    }
    if (!utils.writeOnFile(tempPathFile, contentToWrite)) {
        throw new Error('Unable to write in "' + tempPathFile + '" in the "add" handler');
    }
    if (fileToTest) {
        var instrumentedSource = utils.getInstrumentedSource(tempPathFile);
        if (instrumentedSource === null) {
            throw new Error('Unable to get the instrumented code in "' + tempPathFile + '"');
        }
        if (!utils.writeOnFile(tempPathFile, instrumentedSource)) {
            throw new Error('Unable to write the instrumented in "' + tempPathFile + '" in the "add" handler');
        }
    }
    ret.pathFile = pathFile;
    ret.pathTempFile = tempPathFile;
    ret.fileToTest = fileToTest;
    return ret;
}
function change(pathFile, commonPath, leenaConfig) {
    var ret = {};
    var fileToTest = isFileToTest(pathFile, leenaConfig);
    var tempPathFile;
    var tempPath;
    var contentToWrite;
    tempPathFile = path.normalize([
        leenaConfig.browserSync.webServer.server,
        commonPath
    ].join(path.sep));
    tempPath = path.dirname(tempPathFile);
    contentToWrite = utils.readFile(pathFile);
    if (contentToWrite === null) {
        throw new Error('Unable to read "' + pathFile + '" in the "change" handler');
    }
    if (!utils.writeOnFile(tempPathFile, contentToWrite)) {
        throw new Error('Unable to write in "' + tempPathFile + '" in the "change" handler');
    }
    if (fileToTest) {
        var instrumentedSource = utils.getInstrumentedSource(tempPathFile);
        if (instrumentedSource === null) {
            throw new Error('Unable to get the instrumented code in "' + tempPathFile + '"');
        }
        if (!utils.writeOnFile(tempPathFile, instrumentedSource)) {
            throw new Error('Unable to write the instrumented in "' + tempPathFile + '" in the "change" handler');
        }
    }
    ret.pathFile = pathFile;
    ret.pathTempFile = tempPathFile;
    ret.fileToTest = fileToTest;
    return ret;
}
function unlink(pathFile, commonPath, leenaConfig) {
    var ret = {};
    var tempPathFile;
    var tempPath;
    tempPathFile = path.normalize([
        leenaConfig.browserSync.webServer.server,
        commonPath
    ].join(path.sep));
    tempPath = path.dirname(tempPathFile);
    if (!utils.fileExists(tempPathFile)) {
        throw new Error('Unable to delete "' + tempPathFile + '". It does not exist');
    }
    if (fs.unlinkSync(pathFile) !== undefined) {
        throw new Error('Unable to delete "' + tempPathFile + '"');
    }
    ret.pathFile = pathFile;
    ret.pathTempFile = tempPathFile;
    ret.fileToTest = isFileToTest(pathFile, leenaConfig);
    return ret;
}
function isFileToTest(pathFile, leenaConfig) {
    for (var k = 0; k < leenaConfig.files.length; k++) {
        if (leenaConfig.files[k].originalPath === pathFile) {
            return true;
        }
    }
    return false;
}
