﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;

namespace AutoSecTools.Components
{
    public class CommandLineArgumentSetter
    {
        public string[] Arguments { get; set; }

        public bool CaseSensitive { get; set; }

        public CommandLineArgumentSetter(string[] Arguments)
        {
            this.Arguments = Arguments;
            CaseSensitive = false;
        }

        private Dictionary<string, PropertyInfo> GetArgDictionary(object Target)
        {
            var argDictionary = new Dictionary<string, PropertyInfo>();

            var type = Target.GetType();

            foreach (var p in type.GetProperties())
            {
                var a = p.GetCustomAttributes(typeof(CommandLineArgumentAttribute), true);

                if (!a.Any())
                    continue;

                var name = (a.First() as CommandLineArgumentAttribute).Name;

                if (!CaseSensitive)
                    name = name.ToLower();

                if (argDictionary.ContainsKey(name))
                    throw new InvalidOperationException(string.Format(
                        "Duplicate argument {0}.",
                        name
                    ));

                argDictionary.Add(name, p);
            }

            return argDictionary;
        }

        private void ThrowArgumentException(string Message, params object[] Args)
        {
            throw new ArgumentException(string.Format(Message, Args));
        }

        public void SetProperties(object Target)
        {
            var argDictionary = GetArgDictionary(Target);

            for (int i = 0; i < Arguments.Length; i++)
            {
                var name = Arguments[i];

                if (!CaseSensitive)
                    name = name.ToLower();

                if (!argDictionary.ContainsKey(name))
                    ThrowArgumentException("Invalid argument {0}.", name);

                var propType = argDictionary[name].PropertyType;

                Func<object> getValue = null;

                if (propType == typeof(bool))
                    getValue = () => true;
                else if (++i == Arguments.Length)
                    ThrowArgumentException("No value for argument {0}.", name);
                else if (propType == typeof(string))
                    getValue = () => Arguments[i];
                else if (propType == typeof(int))
                    getValue = () => int.Parse(Arguments[i]);
                else
                    ThrowArgumentException("Invalid property type {0}.", propType);

                argDictionary[name].SetValue(Target, getValue(), null);
            }
        }
    }
}
