/$ Net.h

/*
AUSTIN - A Palm OS Security Scanner.
Copyright (C) 2003  @stake, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

// Handles all Network communications

#ifndef	__NET_H__
#define	__NET_H__

// Socket Type for netSocketOpen()
#define NET_SOCK_STREAM		0
#define	NET_SOCK_DGRAM		1
#define	NET_SOCK_RAW		2

#define	NET_ERROR			-1
#define	NET_NOFLAGS			0

#define NET_DEFAULTTIMEOUT	5

int _Net_tickspersec;

@doc "Sets the timeout value in NetLib, input is in seconds.";
setNetworkTimeout(int t) {
	if (t != -1) t = _Net_tickspersec * t;
	netSetTimeout(t);
}

@doc "Initialize the PalmOS network library.";
initNet() {
	int err;
	
	err = netLibInit();
	if (err != 0) {
		alert("Failed to open network library, AUSTIN will exit. (Error " + err + ")");
		exit();
	}
	_Net_tickspersec = getsysval(4);
	setNetworkTimeout(NET_DEFAULTTIMEOUT);
}

@doc "Close the PalmOS network library.";
clearNet() {
	netLibClose();
}

@doc "Return the error (if any) from the last network call made.";
int netError() {
	return netGetError();
}

@doc "Connects to host and port via TCP. Returns -1 on error, socket ref otherwise.";
int tcpConnect(string host, int port) {
	int sock, res, err;
	
	sock = netSocketOpen(NET_SOCK_STREAM, &err);
	if (sock == -1 || err != 0) {
		writeln("Net.h: tcpConnect() netSocketOpen failed, err=" + hex(err));
		return -1;
	}
	res = netSocketConnect(sock, host, port, &err);
	if (res == -1) {
		writeln("Net.h: tcpConnect() netSocketConnect failed, err=" + hex(err));
		netSocketClose(sock);
		return -1;
	}
	return sock;
}

@doc "Creates UDP-based socket to host and port. Returns -1 on error, socket ref otherwise.";
int udpConnect(string ip, int port) {
	int sock, res, err;
	
	sock = netSocketOpen(NET_SOCK_DGRAM, &err);
	if (sock == -1 || err != 0) {
		writeln("Net.h: udpConnect() netSocketOpen failed, err=" + hex(err));
		return -1;
	}
	res = netSocketConnect(sock, ip, port, &err);
	if (res == -1) {
		writeln("Net.h: udpConnect() netSocketConnect failed, err=" + hex(err));
		netSocketClose(sock);
		return -1;
	}
	return sock;
}

@doc "Close and free resources associated with a TCP socket.";
tcpClose(int sock) {
	netSocketClose(sock);
}

@doc "Close and free resources associated with a UDP socket.";
udpClose(int sock) {
	netSocketClose(sock);
}

@doc "Write data to a socket. Returns number of bytes sent, 0 if socket closed, -1 if error.";
int tcpWrite(int sock, string str) {
	int err;
	
	return netSocketSend(sock, str, strlen(str), NET_NOFLAGS, &err);
}

@doc "Reads the number of bytes specified by len and stores them in buf. Returns number of bytes read, 0 if socket closed, -1 if error.";
int tcpRead(int sock, pointer buf, int len) {
	int err, bytes, totbytes;
	string mybuf;
	
	while (totbytes < len) {
		bytes = netSocketReceive(sock, &mybuf, len, NET_NOFLAGS, &err);
		if (bytes == 0) break;
		else if (bytes == -1) return -1;
		totbytes = totbytes + bytes;
		*buf = *buf + mybuf;
	}
	return totbytes;
}
@doc "Reads until the number of bytes specified by len have been read or token is read. Result is stored in buf. Returns number of bytes read, 0 if socket closed, -1 if error.";
int tcpReadUntil(int sock, pointer buf, int len, string token) {
	int err, bytes, totbytes;
	string mybuf;
	
	while (totbytes < len && strstr(*buf, token, 0) == -1) {
		bytes = netSocketReceive(sock, &mybuf, len, NET_NOFLAGS, &err);
		if (bytes == 0) break;
		else if (bytes == -1) return -1;
		totbytes = totbytes + bytes;
		*buf = *buf + mybuf;
	}
	return totbytes;
}

#endif
