# SPDX-FileCopyrightText: Copyright (c) 2023-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
# SPDX-License-Identifier: LicenseRef-NvidiaProprietary
#
# NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
# property and proprietary rights in and to this material, related
# documentation and any modifications thereto. Any use, reproduction,
# disclosure or distribution of this material and related documentation
# without an express license agreement from NVIDIA CORPORATION or
# its affiliates is strictly prohibited.

import os
from pathlib import Path

from nsys_recipe import nsys_constants
from nsys_recipe.lib import exceptions


def _get_candidate_directories():
    candidate_dirs = []

    parents = Path(__file__).parents
    nsys_dir = os.getenv("NSYS_DIR")

    if nsys_dir is not None:
        nsys_dir = Path(nsys_dir)
        candidate_dirs.append(nsys_dir)

    if len(parents) >= 5:
        # e.g. nsight-systems/target-linux-x64
        nsys_exe_dir = parents[4].resolve()

        if nsys_dir is not None:
            candidate_dirs.append(nsys_dir / nsys_exe_dir.name)
        candidate_dirs.append(nsys_exe_dir)

    # e.g. nsight-systems/target-linux-x64/python/packages
    candidate_dirs.append(parents[2].resolve())

    return candidate_dirs


def find_installed_file(relative_path):
    """Deduce the full path of the given relative path.

    The search order is the following:
    1. Env var NSYS_DIR that points to the Nsys install directory path
    2. Env var NSYS_DIR with target directory name, deduced from parent
    3. Target directory, deduced from parent
    4. Root package directory that contains the 'nsys_recipe' package
    """
    candidate_dirs = _get_candidate_directories()

    for candidate_dir in candidate_dirs:
        candidate = candidate_dir / relative_path

        if candidate.exists():
            return str(candidate)

    raise exceptions.ValueError(
        f"Cannot find '{relative_path}'."
        " Please set NSYS_DIR to a valid Nsys install directory that contains internal dependencies."
    )
